/* -LICENSE-START-
 ** Copyright (c) 2012 Blackmagic Design
 **
 ** Permission is hereby granted, free of charge, to any person or organization
 ** obtaining a copy of the software and accompanying documentation covered by
 ** this license (the "Software") to use, reproduce, display, distribute,
 ** execute, and transmit the Software, and to prepare derivative works of the
 ** Software, and to permit third-parties to whom the Software is furnished to
 ** do so, all subject to the following:
 **
 ** The copyright notices in the Software and this entire statement, including
 ** the above license grant, this restriction and the following disclaimer,
 ** must be included in all copies of the Software, in whole or in part, and
 ** all derivative works of the Software, unless such copies or derivative
 ** works are solely in the form of machine-executable object code generated by
 ** a source language processor.
 **
 ** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 ** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 ** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 ** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 ** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 ** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 ** DEALINGS IN THE SOFTWARE.
 ** -LICENSE-END-
 */

#ifndef __OPENGL_COMPOSITE_H__
#define __OPENGL_COMPOSITE_H__

#include <windows.h>
#include <process.h>
#include <tchar.h>
#include <gl/gl.h>
#include <gl/glu.h>

#include <objbase.h>
#include <comutil.h>
#include "DeckLinkAPI_h.h"

#include "VideoFrameTransfer.h"

#include <map>
#include <vector>
#include <deque>

class PlayoutDelegate;
class CaptureDelegate;
class PinnedMemoryAllocator;


class OpenGLComposite
{
public:
	OpenGLComposite(HWND hWnd, HDC hDC, HGLRC hRC);
	~OpenGLComposite();

	bool InitDeckLink();
	bool Start();
	bool Stop();

	void resizeGL(WORD width, WORD height);
	void paintGL();

	void VideoFrameArrived(IDeckLinkVideoInputFrame* inputFrame, bool hasNoInputSource);
	void PlayoutFrameCompleted(IDeckLinkVideoFrame* completedFrame, BMDOutputFrameCompletionResult result);

private:
	void resizeWindow(int width, int height);
	bool CheckOpenGLExtensions();

	CaptureDelegate*						mCaptureDelegate;
	PlayoutDelegate*						mPlayoutDelegate;
	HWND									hGLWnd;
	HDC										hGLDC;
	HGLRC									hGLRC;
	CRITICAL_SECTION						pMutex;

	// DeckLink
	IDeckLinkInput*							mDLInput;
	IDeckLinkOutput*						mDLOutput;
	std::deque<IDeckLinkMutableVideoFrame*>	mDLOutputVideoFrameQueue;
	PinnedMemoryAllocator*					mCaptureAllocator;
	PinnedMemoryAllocator*					mPlayoutAllocator;
	BMDTimeValue							mFrameDuration;
	BMDTimeScale							mFrameTimescale;
	unsigned								mTotalPlayoutFrames;
	unsigned								mFrameWidth;
	unsigned								mFrameHeight;
	bool									mHasNoInputSource;

	// OpenGL data
	bool									mFastTransferExtensionAvailable;
	GLuint									mCaptureTexture;
	GLuint									mFBOTexture;
	GLuint									mUnpinnedTextureBuffer;
	GLuint									mIdFrameBuf;
	GLuint									mIdColorBuf;
	GLuint									mIdDepthBuf;
	GLuint									mProgram;
	GLuint									mFragmentShader;
	GLfloat									mRotateAngle;
	GLfloat									mRotateAngleRate;
	int										mViewWidth;
	int										mViewHeight;

	bool InitOpenGLState();
	bool compileFragmentShader(int errorMessageSize, char* errorMessage);
};

////////////////////////////////////////////
// PinnedMemoryAllocator
////////////////////////////////////////////
class PinnedMemoryAllocator : public IDeckLinkMemoryAllocator
{
public:
	PinnedMemoryAllocator(HDC hdc, HGLRC hglrc, VideoFrameTransfer::Direction direction, unsigned cacheSize);
	virtual ~PinnedMemoryAllocator();

	bool transferFrame(void* address, GLuint gpuTexture);
	void waitForTransferComplete(void* address);
	void beginTextureInUse();
	void endTextureInUse();
	void unPinAddress(void* address);

	// IUnknown methods
	virtual HRESULT STDMETHODCALLTYPE	QueryInterface(REFIID iid, LPVOID *ppv);
	virtual ULONG STDMETHODCALLTYPE		AddRef(void);
	virtual ULONG STDMETHODCALLTYPE		Release(void);

	// IDeckLinkMemoryAllocator methods
	virtual HRESULT STDMETHODCALLTYPE	AllocateBuffer (unsigned int bufferSize, void* *allocatedBuffer);
	virtual HRESULT STDMETHODCALLTYPE	ReleaseBuffer (void* buffer);
	virtual HRESULT STDMETHODCALLTYPE	Commit ();
	virtual HRESULT STDMETHODCALLTYPE	Decommit ();

private:
	HDC											mHGLDC;
	HGLRC										mHGLRC;
	LONG										mRefCount;
	VideoFrameTransfer::Direction				mDirection;
	std::map<void*, VideoFrameTransfer*>		mFrameTransfer;
	std::map<void*, unsigned long>				mAllocatedSize;
	std::vector<void*>							mFrameCache;
	unsigned									mFrameCacheSize;
};

////////////////////////////////////////////
// Capture Delegate Class
////////////////////////////////////////////

class CaptureDelegate : public IDeckLinkInputCallback
{
	OpenGLComposite*	m_pOwner;
	LONG				mRefCount;

public:
	CaptureDelegate (OpenGLComposite* pOwner);

	// IUnknown needs only a dummy implementation
	virtual HRESULT	STDMETHODCALLTYPE	QueryInterface (REFIID iid, LPVOID *ppv);
	virtual ULONG	STDMETHODCALLTYPE	AddRef ();
	virtual ULONG	STDMETHODCALLTYPE	Release ();

	virtual HRESULT STDMETHODCALLTYPE	VideoInputFrameArrived (IDeckLinkVideoInputFrame *videoFrame, IDeckLinkAudioInputPacket *audioPacket);
	virtual HRESULT	STDMETHODCALLTYPE	VideoInputFormatChanged (BMDVideoInputFormatChangedEvents notificationEvents, IDeckLinkDisplayMode *newDisplayMode, BMDDetectedVideoInputFormatFlags detectedSignalFlags);
};

////////////////////////////////////////////
// Render Delegate Class
////////////////////////////////////////////

class PlayoutDelegate : public IDeckLinkVideoOutputCallback
{
	OpenGLComposite*	m_pOwner;
	LONG				mRefCount;

public:
	PlayoutDelegate (OpenGLComposite* pOwner);

	// IUnknown needs only a dummy implementation
	virtual HRESULT	STDMETHODCALLTYPE	QueryInterface (REFIID iid, LPVOID *ppv);
	virtual ULONG	STDMETHODCALLTYPE	AddRef ();
	virtual ULONG	STDMETHODCALLTYPE	Release ();

	virtual HRESULT	STDMETHODCALLTYPE	ScheduledFrameCompleted (IDeckLinkVideoFrame* completedFrame, BMDOutputFrameCompletionResult result);
	virtual HRESULT	STDMETHODCALLTYPE	ScheduledPlaybackHasStopped ();
};

#endif		// __OPENGL_COMPOSITE_H__
